unit IWHTMLControls;

{PUBDIST}

interface

uses
  {$IFDEF VSNET}
  System.ComponentModel, System.Drawing, IWNetClasses, IWURLTargetConverter,
  IWNETBaseControl,
  {$ELSE}
    {$IFDEF Linux}
    IWCLXComponent, IWCLXClasses,
    {$ELSE}
    IWVCLComponent, IWVCLClasses,
    {$ENDIF}
  {$ENDIF}
  Classes, IWHTMLTag, IWScriptEvents, IWControl, IWGlobal, IWColor,
  IWRenderContext, IWBaseInterfaces;

type
  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWHRule.bmp}
  TIWHRule = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWHRule), 'TIWHRule.bmp')]
  {$ENDIF}
  TIWHRule = class(TIWCustomControl)
  //TODO: add a percentage prop
  protected
    procedure InitControl; override;
    function get_HasTabOrder: Boolean; override;
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
    function RenderStyle(AContext: TIWBaseHTMLComponentContext): string; override;
  published
    property FriendlyName;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWList.bmp}
  TIWList = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWList), 'TIWList.bmp')]
  {$ENDIF}
  TIWList = class(TIWCustomControl)
  protected
    FNumbered: Boolean;
    FItems: TIWStringList;
    //
    procedure OnItemsChange( ASender : TObject );
    procedure SetItems(AValue: TIWStringList);
    procedure SetNumbered(const AValue: Boolean);

    procedure InitControl; override;
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    procedure Clear; virtual;
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
  published
    property FriendlyName;
    {$IFDEF CLR}
    property WebFont;
    {$ELSE}
    property Font;
    {$ENDIF}
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    {$ENDIF}
    property Items: TIWStringList read FItems write SetItems;
    property Numbered: boolean read FNumbered write SetNumbered;
  end;

  TIWLinkBase = class(TIWCustomControl, IIWSubmitControl)
  protected
    FAlignment: TAlignment;
    FSubmitParam : String;
    FRawText: Boolean;

    {$IFDEF VSNET}
    function getText: string; reintroduce;virtual;
    procedure setText(AValue: string); reintroduce;virtual;
    {$ENDIF}
    procedure InitControl; override;
    procedure Submit(const AValue: string); override;
    procedure SetAlignment(const Value: TAlignment);
  public
    function RenderStyle(AComponentContext: TIWBaseHTMLComponentContext): string; override;
    function GetSubmitParam : String;

    property RawText: Boolean read FRawText write FRawText;
    {$IFDEF VSNET}
    property Text: string read getText write setText;
    {$ENDIF}
  published
    property Alignment: TAlignment read FAlignment write SetAlignment;

    property Enabled;
    property ExtraTagParams;
    {$IFDEF CLR}
    property WebColor;
    property WebFont;
    {$ELSE}
    property Color;
    property Font;
    {$ENDIF}
    property ScriptEvents;
  end;

  {$IFDEF VSNET}
  [TypeConverter(typeof(TIWURLTargetConverter))]
  {$ENDIF}
  TIWURLTarget = class(TPersistent)
  protected
    FAddressBar: Boolean;
    FHeight: integer;
    FLeft: integer;
    FMenu: Boolean;
    FResizable: boolean;
    FScrollbars: boolean;
    FToolbar: Boolean;
    FTop: integer;
    FWidth: integer;
    FWindowName: string;
  public
    constructor Create; overload;
    constructor Create(AAddressBar: Boolean; AHeight: integer;
      ALeft: Integer; AMenu: Boolean; AResizable: Boolean;
      AScrollbars: Boolean; AToolbar: Boolean; ATop: Integer;
      AWidth: Integer; AWindowName: string); overload;
  published
    property AddressBar: Boolean read FAddressBar write FAddressBar;
    property Height: Integer read FHeight write FHeight;
    property Left: Integer read FLeft write FLeft;
    property Menu: Boolean read FMenu write FMenu;
    property Resizable: Boolean read FResizable write FResizable;
    property Scrollbars: boolean read FScrollbars write FScrollbars;
    property Toolbar: Boolean read FToolbar write FToolbar;
    property Top: Integer read FTop write FTop;
    property Width: Integer read FWidth write FWidth;
    property WindowName: string read FWindowName write FWindowName;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWLink.bmp}
  TIWLink = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWLink), 'TIWLink.bmp')]
  {$ENDIF}
  TIWLink = class(TIWLinkBase)
  protected
    procedure InitControl; override;
    procedure HookEvents(AContext: TIWPageContext40; AScriptEvents: TIWScriptEvents); override;
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
  published
    property Confirmation;
    property DoSubmitValidation;
    property FriendlyName;

    property OnClick;
    property TabOrder;
    property RawText;
    {$IFDEF VSNET}
    property Text;
    {$ELSE}
    property Caption;
    {$ENDIF}
  end;

  TIWCustomURL = class(TIWLinkBase)
  protected
    FTargetOptions: TIWURLTarget;
    FTerminateApp: boolean;
    FUseTarget: boolean;
    FURL: string;
    //
    procedure SetTerminateApp(const AValue: boolean);
    procedure SetUseTarget(const AValue: boolean);
    procedure HookEvents(AContext: TIWPageContext40; AScriptEvents: TIWScriptEvents); override;

    procedure InitControl; override;
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
  published
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    {$ENDIF}
    property TargetOptions: TIWURLTarget read FTargetOptions write FTargetOptions;
    property TerminateApp: boolean read FTerminateApp write SetTerminateApp;
    property UseTarget: boolean read FUseTarget write SetUseTarget;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWURL.bmp}
  TIWURL = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWURL), 'TIWURL.bmp')]
  {$ENDIF}
  TIWURL = class(TIWCustomURL)
  published
    property FriendlyName;
    {$IFDEF VSNET}
    [DefaultValue('')]
    {$ENDIF}
    property URL: string read FURL write FURL;
    property TabOrder;
    property RawText;
    {$IFDEF VSNET}
    property Text;
    {$ELSE}
    property Caption;
    {$ENDIF}
  end;

  TIWAppletAlignment = (aaTop, aaMiddle, aaBottom, aaLeft, aaRight);

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWApplet.bmp}
  TIWApplet = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWApplet), 'TIWApplet.bmp')]
  {$ENDIF}
  TIWApplet = class(TIWCustomControl)
  protected
    FAlignment: TIWAppletAlignment;
    FAltText: string;
    FAppletName: string;
    FArchive: string;
    FClassFile: string;
    FCodeBase: string;
    FHorizSpace: Integer;
    FParams: TIWStringList;
    FVertSpace: Integer;
    //
    procedure SetParams(const AValue: TIWStringList);
    procedure InitControl; override;
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
  published
    property Alignment: TIWAppletAlignment read FAlignment write FAlignment;
    {$IFDEF VSNET}
    [DefaultValue('')]
    {$ENDIF}
    property AltText: string read FAltText write FAltText;
    {$IFDEF VSNET}
    [DefaultValue('')]
    {$ENDIF}
    property AppletName: string read FAppletName write FAppletName;
    {$IFDEF VSNET}
    [DefaultValue('')]
    {$ENDIF}
    property Archive: string read FArchive write FArchive;
    {$IFDEF VSNET}
    [DefaultValue('')]
    {$ENDIF}
    property ClassFile: string read FClassFile write FClassFile;
    {$IFDEF VSNET}
    [DefaultValue('')]
    {$ENDIF}
    property CodeBase: string read FCodeBase write FCodeBase;
    property FriendlyName;
    property HorizSpace: Integer read FHorizSpace write FHorizSpace;
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    {$ENDIF}
    property Params: TIWStringList read FParams write SetParams;
    property VertSpace: Integer read FVertSpace write FVertSpace;
  end;

  TIWURLWindowScrolling = (usYes, usNo, usAuto);

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWURLWindow.bmp}
  TIWURLWindow = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWURLWindow), 'TIWURLWindow.bmp')]
  {$ENDIF}
  TIWURLWindow = class(TIWCustomControl)
  protected
    FURI: string;
    FLongDesc: string;
    FBorder: Boolean;
    FMarginWidth: Integer;
    FMarginHeight: Integer;
    FScrolling: TIWURLWindowScrolling;

    procedure InitControl; override;
    procedure SetURI(AValue: string);
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;
  published
    property FriendlyName;
    property Align;
    {$IFDEF VSNET}
    [DefaultValue('')]
    {$ENDIF}
    property URI: string read FURI write SetURI;
    {$IFDEF VSNET}
    [DefaultValue('')]
    {$ENDIF}
    property LongDesc: string read FLongDesc write FLongDesc;
    property Border: Boolean read FBorder write FBorder;
    property MarginWidth: Integer read FMarginWidth write FMarginWidth;
    property MarginHeight: Integer read FMarginHeight write FMarginHeight;
    property Scrolling: TIWURLWindowScrolling read FScrolling write FScrolling;
    property Height;
    property Width;
  end;


implementation

uses
  {$IFDEF Linux}QGraphics, {$ELSE}Graphics, {$ENDIF}
  IWBaseControl, IWCompLabel, IWFont,
  IWHTML40Interfaces, IWMarkupLanguageTag, IWTypes,
  {$IFNDEF VSNET}Controls,{$ENDIF}
  SWSystem, SysUtils;

{$IFDEF VSNET}
function TIWLinkBase.getText: string;
begin
  Result := Caption;
end;

procedure TIWLinkBase.setText(AValue: string);
begin
  Caption := AValue;
  TIWNETBaseControl(Self).Invalidate;
end;
{$ENDIF}

{ TIWHRule }

procedure TIWHRule.InitControl;
begin
  inherited;
  Width := 100;
  Height := 3;
  RenderSize := True;
end;


function TIWHRule.get_HasTabOrder: Boolean;
begin
  Result := False;
end;

function TIWHRule.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
begin
  Result := TIWHTMLTag.CreateTag('HR'); try
    if RenderSize then begin
      Result.AddIntegerParam('WIDTH', Width);
      Result.AddIntegerParam('HEIGHT', Height);
    end;
  except FreeAndNil(Result); raise; end;
end;

function TIWHRule.RenderStyle(AContext: TIWBaseHTMLComponentContext): string;
begin
  Result := inherited RenderStyle(AContext);
  // following is to prevent FF from displaying the HRule lower than established
  // at design time
  if AContext.Browser = brNetscape7 then begin
    Result := Result + 'margin:0;';
  end;

end;


{ TIWList }

procedure TIWList.InitControl;
begin
  inherited;
  FItems := TIWStringList.Create;
  FItems.OnChange := OnItemsChange;  
  Height := 24;
  Width := 116;
end;

procedure TIWList.OnItemsChange( ASender : TObject );
begin
  DoRefreshControl := True;
  Invalidate;  
end;

procedure TIWList.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FItems);
  inherited;
end;

function TIWList.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  i: Integer;
begin
  Result := nil;
  if Items.Count > 0 then begin
    Result := TIWHTMLTag.CreateTag('table');
    try
      for i := 0 to Items.Count - 1 do begin
        with Result.Contents.AddTag('TR') do begin
          with Contents.AddTag('TD') do begin
            if Numbered then begin
              Contents.AddText(IntToStr(i + 1) + '.');
            end else begin
              Contents.AddText(Chr(149));
            end;
          end;
          with Contents.AddTag('TD') do begin
            Contents.AddText(Items.Strings[i]);
          end;
        end;
      end;
      if Result.Contents.Count = 0 then
        Result.Contents.AddText('');
    except
      FreeAndNil(Result);
      raise;
    end;
  end;
end;

procedure TIWList.SetItems(AValue: TIWStringList);
begin
  FItems.Assign(AValue);
  Invalidate;
end;

procedure TIWList.SetNumbered(const AValue: Boolean);
begin
  FNumbered := AValue;
  Invalidate;
end;

procedure TIWList.Clear;
begin
  Items.Clear;
end;

{ TIWLink }

procedure TIWLink.InitControl;
begin
  inherited;
  // Default to false for links.
  DoSubmitValidation := False;
  // Default RenderSize to true
  RenderSize := true;
end;

procedure TIWLink.HookEvents(AContext: TIWPageContext40; AScriptEvents: TIWScriptEvents);
var
  LSubmitHandler: string;
begin
  inherited HookEvents(AContext, AScriptEvents);

  if (RenderSize) and (Alignment <> taLeftJustify) then begin
    LSubmitHandler := Format('if (document.getElementById(''%s'').childNodes[0].href) {%s} else return false;', [HTMLName, iif(HasOnClick, SubmitHandler('', '', true))]);
  end else begin
    LSubmitHandler := Format('if (document.getElementById(''%s'').href) {%s} else return false;', [HTMLName, iif(HasOnClick, SubmitHandler('', '', true))]);
  end;
  
  AScriptEvents.HookEvent('OnClick', LSubmitHandler);
end;

function TIWLink.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  LLabel: TIWLabel;
  LTempTag : TIWHTMLTag;
begin
  if Enabled and HasOnClick then
  begin
    // Render in a DIV only in RenderSize and Alignment <> taLeftJustify
    if (RenderSize) and (Alignment <> taLeftJustify) then
    begin
      Result := TIWHTMLTag.CreateTag('DIV');
      case Alignment of
        taCenter: Result.AddStringParam('style', 'text-align:center;');
        taRightJustify: Result.AddStringParam('style', 'text-align:right;');
      end;
      LTempTag := Result;
      Result := Result.Contents.AddTag('A');
    end
    else
    begin
      Result := TIWHTMLTag.CreateTag('A');
      LTempTag := Result;
    end;

    try
      if AContext.WebApplication.Browser = brIE then
      begin
        Result.AddStringParam('HREF', HTMLName);
      end
      else
        if AContext.WebApplication.Browser = brOpera then
        begin
          Result.AddStringParam('HREF', '#');
        end
        else
        begin
          Result.AddStringParam('HREF', 'javascript:void(0);');
        end;

      if (RenderSize) and (Alignment <> taLeftJustify) then
      begin
        Result.AddStringParam('style', WebFont.FontToStringStyle(AContext.WebApplication.Browser));
      end;


      if IsDesignMode then
      begin
        Result.AddStringParam('onclick', 'return false;');
      end;

      if not FRawText then
      begin
        Result.Contents.AddText(TextToHTML(Caption));
      end
      else
      begin
        Result.Contents.AddText(Caption);
      end;
      Result := LTempTag;
    except
      FreeAndNil(Result);
      raise;
    end;
  end
  else
  begin
    LLabel := TIWLabel.Create{$IFNDEF VSNET}(Self){$ENDIF};
    try
      LLabel.Name := HTMLName;
      LLabel.RawText := FRawText;
      LLabel.Caption := Caption;
      LLabel.Alignment := Alignment;
      LLabel.WebFont.Assign(WebFont);
      LLabel.AutoSize := false;
      LLabel.Width := Width;
      LLabel.Height := Height;
      LLabel.Visible := true;
      LLabel.RenderSize := true;
      Result := LLabel.RenderHTML(AContext);
    finally
      FreeAndNil(LLabel);
    end;
  end;
end;

{ TIWLinkBase }

procedure TIWLinkBase.SetAlignment(const Value: TAlignment);
begin
  if FAlignment <> Value then
  begin
    FAlignment := Value;
    Invalidate;
  end;
end;

function TIWLinkBase.GetSubmitParam: String;
begin
  Result := FSubmitParam;
end;

procedure TIWLinkBase.InitControl;
begin
  inherited;
  Width := 65;
  Height := 17;
  WebColor := fromTColor(clNone);
  WebFont.Style := [{$IFDEF VSNET}TIWFontStyle.{$ENDIF}fsUnderline];
end;

procedure TIWLinkBase.Submit(const AValue: string);
begin
  FSubmitParam := AValue;
  DoClick;
end;

function TIWLinkBase.RenderStyle(AComponentContext: TIWBaseHTMLComponentContext): string;
begin
  result := inherited RenderStyle(AComponentContext) + iif(toTColor(WebColor) <> clNone, 'background-Color: ' + ColorToRGBString(WebColor) + ';');
end;

{ TIWURLTarget }

constructor TIWURLTarget.Create;
begin
  inherited;
  // These are the target window coords, not TIWRURLTarget's
  Left := -1;
  Top := -1;
  Width := -1;
  Height := -1;
  Resizable := True;
  ScrollBars := True;
  Menu := False;
  AddressBar := False;
end;

{ TIWCustomURL }

procedure TIWCustomURL.InitControl;
begin
  inherited;
  FTargetOptions := TIWURLTarget.Create;
  FURL := '';
end;

procedure TIWCustomURL.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FTargetOptions);
  inherited;
end;

procedure TIWCustomURL.HookEvents(AContext: TIWPageContext40; AScriptEvents: TIWScriptEvents);
begin
  inherited HookEvents(AContext, AScriptEvents);
  if Enabled then begin
  // TODO: Move OnClick code here
  end;
end;

function TIWCustomURL.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  LURL: string;
  LResult: String;
  LTempTag : TIWHTMLTag;
begin
  if not Enabled then
  begin
    Result := TIWHTMLTag.CreateTag('SPAN');
    try
      if not FRawText then
      begin
        Result.Contents.AddText(TextToHTML(Caption));
      end
      else
      begin
        Result.Contents.AddText(Caption);
      end;
    except
      FreeAndNil(Result);
      raise;
    end;
  end
  else
  begin
    Result := TIWHTMLTag.CreateTag('A');
    try
      // other than http or https? IE5 doesnt like mailto: (and possibly others) in LoadURL
      if not (AnsiSameText(Copy(FURL, 1, 7), 'http://') or AnsiSameText(Copy(FURL, 1, 8), 'https://')) and not
        (Copy(FURL, 1, 1)='.') and not (Copy(FURL, 1, 1)='/') then
      begin
          Result.AddStringParam('HREF', FURL);
      end
      else
      begin
        Result.AddStringParam('HREF', '#' + HTMLName);
        if UseTarget then
        begin
          LResult := '';
          with TargetOptions do
          begin
            // personalbar
            // menubar
            // location - If yes, creates a Location entry field.
            // status - statusbar
            LResult := LResult + ',location=' + iif(AddressBar, 'yes', 'no');
            LResult := LResult + ',menubar=' + iif(Menu, 'yes', 'no');
            LResult := LResult + ',resizable=' + iif(Resizable, 'yes', 'no');
            LResult := LResult + ',toolbar=' + iif(Toolbar, 'yes', 'no');
            LResult := LResult + ',scrollbars=' + iif(Scrollbars, 'yes', 'no');

            if Left > -1 then
            begin
              LResult := LResult + ',left=' + IntToStr(Left);
            end;

            if Top > -1 then
            begin
              LResult := LResult + ',top=' + IntToStr(Top);
            end;

            if Width > -1 then
            begin
              LResult := LResult + ',width=' + IntToStr(Width);
            end;

            if Height > -1 then
            begin
              LResult := LResult + ',height=' + IntToStr(Height);
            end;
          end;

          Result.AddStringParam('OnClick', 'NewWindow(''' + FURL + ''', ''' + TargetOptions.WindowName
            + ''',''' + Copy(LResult, 2, MaxInt) + ''')');
        end
        else
        begin
          LURL := FURL;
          if TerminateApp then
          begin
            if AnsiSameText(Copy(LURL, 1, 7), 'http://') then
            begin
              Delete(LURL, 1, 7);
            end
            else
              if AnsiSameText(Copy(LURL, 1, 8), 'https://') then
              begin
                Delete(LURL, 1, 8);
                LURL := '_s_.' + LURL;
              end;
            LURL := '/STOP/' + AContext.WebApplication.AppID + '/' + LURL;
          end;
          Result.AddStringParam('OnClick', 'parent.LoadURL(''' + LURL + ''')');
        end;
      end;
      if Hint = '' then
      begin
        Hint := FURL;
      end;
      HintEvents(Result);
      if not FRawText then
      begin
        Result.Contents.AddText(TextToHTML(Caption));
      end
      else
      begin
        Result.Contents.AddText(Caption);
      end;
    except
      FreeAndNil(Result);
      raise;
    end;

    // Render in a DIV only in RenderSize and Alignment <> taLeftJustify
    if (RenderSize) and (Alignment <> taLeftJustify) then
    begin
      LTempTag := Result;
      Result := TIWHTMLTag.CreateTag('DIV');
      try
        with Result do
        begin
          case Alignment of
            taCenter: Result.AddStringParam('style', 'text-align:center;');
            taRightJustify: Result.AddStringParam('style', 'text-align:right;');
          end;
          Contents.AddTagAsObject(LTempTag);
        end;
      except
        FreeAndNil(Result);
        raise;
      end;
    end;
  end;
end;

procedure TIWCustomURL.SetTerminateApp(const AValue: boolean);
begin
  FTerminateApp := AValue;
  if AValue then begin
    FUseTarget := False;
  end;
end;

procedure TIWCustomURL.SetUseTarget(const AValue: boolean);
begin
  FUseTarget := AValue;
  if AValue then begin
    FTerminateApp := False;
  end;
end;

{ TIWApplet }

procedure TIWApplet.InitControl;
begin
  inherited;
  FParams := TIWStringList.Create;
  Height := 200;
  Width := 200;
  FAltText := '';
  FAppletName := '';
  FArchive := '';
  FClassFile := '';
  FCodeBase := '';
end;

procedure TIWApplet.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FParams);
  inherited;
end;

function TIWApplet.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  i: Integer;
  LAlign: string;
begin
  case Alignment of
    aaTop: LAlign := 'top';
    aaMiddle: LAlign := 'middle';
    aaBottom: LAlign := 'bottom';
    aaLeft: LAlign := 'left';
    aaRight: LAlign := 'right';
  end;
  Result := TIWHTMLTag.CreateTag('APPLET'); try
    Result.AddStringParam('CODE', ClassFile);
    Result.AddIntegerParam('HEIGHT', Height);
    Result.AddIntegerParam('WIDTH', Width);
    Result.AddStringParam('ALIGN', LAlign);
    Result.AddStringParam('CODEBASE', iif(CodeBase, CodeBase, GServerController.FilesURL));
    Result.Add(iif(AppletName, ' NAME=' + AppletName));
    Result.Add(iif(Width > 0, ' HSPACE=' + IntToStr(HorizSpace)));
    Result.Add(iif(Height > 0, ' VSPACE=' + IntToStr(VertSpace)));
    Result.Add(iif(AltText, ' ALT=' + AltText + ''));
    Result.Add(iif(Archive, ' ARCHIVE=' + Archive + ''));

    for i := 0 to Params.Count - 1 do begin
      with Result.Contents.AddTag('PARAM') do begin
        AddStringParam('NAME', Self.Params.Names[i]);
        AddStringParam('VALUE', Self.Params.Values[Self.Params.Names[i]]);
      end;
    end;
  except FreeAndNil(Result); raise; end;
end;

procedure TIWApplet.SetParams(const AValue: TIWStringList);
begin
  FParams.Assign(AValue);
end;

{ TIWURLWindow }

procedure TIWURLWindow.InitControl;
begin
  inherited;
  FScrolling := usYes;
  Height := 100;
  Width := 150;
  FURI := '';
  FLongDesc := '';
end;

procedure TIWURLWindow.SetURI(AValue: string);
begin
  if FURI <> AValue then
  begin
    DoRefreshControl := true;
    FURI := AValue;
    Invalidate;
  end;
end;

function TIWURLWindow.RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
var
  LParent : TPlatformControl;

  function FindParent(AComponent : TPlatformControl) : TPlatformControl;
  begin
    if SupportsInterface(AComponent.Parent, IIWHTML40Form) then
    begin
      Result := AComponent.Parent;
    end
    else
    begin
      if AComponent.Parent <> nil then
      begin
        Result := FindParent(AComponent.Parent);
      end
      else
      begin
        Result := nil;
      end;
    end;
  end;
begin
  Result := TIWHTMLTag.CreateHTMLTag('IFRAME', cbTrue);
  try
    // The frame source is set via the init proc in umAll becase IE gives up on executing the
    // Javascript code in the page if an invalid URL is specified for an IFRAME.
    // This fixes BBG #741
    with Result do
    begin
      LParent := FindParent(Self);
      if LParent <> nil then
      begin
        if HTML40FormInterface(LParent).PageContext.UpdateMode = umPartial then
        begin
          AddStringParam('SRC', FURI);
        end
        else
        begin
          if FURI <> '' then
          begin
            (AContext as TIWComponent40Context).AddToInitProc(Format('frames[''%s''].location.href=''%s''', [HTMLName, FURI]));
          end;
        end;
      end;
      AddStringParam('LONGDESC', FLongDesc);
      // Always render size as a fix for BBG #903 - IWURLWindow - wrong position and size
      AddIntegerParam('WIDTH', Width);
      AddIntegerParam('HEIGHT', Height);
      AddIntegerParam('FRAMEBORDER', StrToInt(IIF(FBorder, '1', '0')));
      AddIntegerParam('MARGINWIDTH', FMarginWidth);
      AddIntegerParam('MARGINHEIGHT', FMarginHeight);
      AddStringParam('NAME', HTMLName);
      case FScrolling of
        usYes: Add('SCROLLING=YES');
        usNo: Add('SCROLLING=NO');
        usAuto: Add('SCROLLING=AUTO');
      end;
    end;

  except
    FreeAndNil(Result);
    raise;
  end;
end;


constructor TIWURLTarget.Create(AAddressBar: Boolean; AHeight: integer;
      ALEft: Integer; AMenu: Boolean; AResizable: Boolean;
      AScrollbars: Boolean; AToolbar: Boolean; ATop: Integer;
      AWidth: Integer; AWindowName: string);
begin
  inherited Create;
  FAddressBar := AAddressBar;
  FHeight:= AHeight;
  FLeft:= ALeft;
  FMenu := AMenu;
  FResizable := AResizable;
  FScrollbars := AScrollbars;
  FToolbar := AToolbar;
  FTop := ATop;
  FWidth := AWidth;
  FWindowName := AWindowName;

end;

end.
